@extends('layouts.app')

@section('title', 'Deposit - ' . config('app.name'))

@section('content')
<div class="container mx-auto px-4 py-8">
    <div class="max-w-4xl mx-auto">
        <!-- Page Header -->
        <div class="mb-8">
            <div class="flex items-center mb-4">
                <a href="{{ route('wallet.index') }}" class="text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 mr-3">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                    </svg>
                </a>
                <h1 class="text-3xl font-bold text-gray-900 dark:text-white">Deposit Funds</h1>
            </div>
            <p class="text-gray-600 dark:text-gray-400">Add money to your wallet using secure payment methods</p>
        </div>

        <!-- Current Balance -->
        <div class="bg-white dark:bg-gray-800 rounded-lg p-6 mb-8 shadow-sm border border-gray-200 dark:border-gray-700">
            <div class="flex items-center justify-between">
                <div>
                    <h2 class="text-lg font-semibold text-gray-900 dark:text-white mb-2">Current Balance</h2>
                    <div class="text-3xl font-bold text-blue-600 dark:text-blue-400">${{ number_format(getUserWallet($user->id), 2) }}</div>
                </div>
                <div class="bg-blue-100 dark:bg-blue-900 rounded-full p-3">
                    <svg class="w-8 h-8 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8c-1.657 0-3 .895-3 2s1.343 2 3 2 3 .895 3 2-1.343 2-3 2m0-8c1.11 0 2.08.402 2.599 1M12 8V7m0 1v8m0 0v1m0-1c-1.11 0-2.08-.402-2.599-1"></path>
                    </svg>
                </div>
            </div>
        </div>

        <!-- Deposit Form -->
        <div class="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Choose Payment Method</h3>
            </div>
            
            <div class="p-6">
                <!-- Payment Method Selection -->
                <div class="mb-6">
                    <label class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-3">Select Payment Method</label>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <label class="relative flex cursor-pointer rounded-lg border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 p-4 shadow-sm focus:outline-none">
                            <input type="radio" name="payment_method" value="stripe" class="sr-only" checked>
                            <span class="flex flex-1">
                                <span class="flex flex-col">
                                    <span class="block text-sm font-medium text-gray-900 dark:text-white">Credit/Debit Card</span>
                                    <span class="mt-1 flex items-center text-sm text-gray-500 dark:text-gray-400">
                                        <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 10h18M7 15h1m4 0h1m-7 4h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3v8a3 3 0 003 3z"></path>
                                        </svg>
                                        Secure payment via Stripe
                                    </span>
                                </span>
                            </span>
                            <svg class="h-5 w-5 text-blue-600" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                        </label>

                        <label class="relative flex cursor-pointer rounded-lg border border-gray-300 dark:border-gray-600 bg-white dark:bg-gray-700 p-4 shadow-sm focus:outline-none">
                            <input type="radio" name="payment_method" value="paypal" class="sr-only">
                            <span class="flex flex-1">
                                <span class="flex flex-col">
                                    <span class="block text-sm font-medium text-gray-900 dark:text-white">PayPal</span>
                                    <span class="mt-1 flex items-center text-sm text-gray-500 dark:text-gray-400">
                                        <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 9V7a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2m2 4h10a2 2 0 002-2v-6a2 2 0 00-2-2H9a2 2 0 00-2 2v6a2 2 0 002 2zm7-5a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                        </svg>
                                        Pay with your PayPal account
                                    </span>
                                </span>
                            </span>
                            <svg class="h-5 w-5 text-gray-400" viewBox="0 0 20 20" fill="currentColor">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                        </label>
                    </div>
                </div>

                <!-- Amount Input -->
                <div class="mb-6">
                    <label for="amount" class="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">Amount to Deposit</label>
                    <div class="relative">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <span class="text-gray-500 dark:text-gray-400 sm:text-sm">$</span>
                        </div>
                        <input type="number" id="amount" name="amount" min="1" step="0.01" 
                               class="block w-full pl-7 pr-12 py-3 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white" 
                               placeholder="0.00" required>
                        <div class="absolute inset-y-0 right-0 pr-3 flex items-center pointer-events-none">
                            <span class="text-gray-500 dark:text-gray-400 sm:text-sm">USD</span>
                        </div>
                    </div>
                    <p class="mt-2 text-sm text-gray-500 dark:text-gray-400">Minimum deposit: $1.00</p>
                </div>

                <!-- Stripe Payment Form -->
                <div id="stripe-form" class="mb-6">
                    <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                        <h4 class="text-sm font-medium text-gray-900 dark:text-white mb-3">Card Details</h4>
                        <div id="card-element" class="border border-gray-300 dark:border-gray-600 rounded-lg p-3 bg-white dark:bg-gray-600">
                            <!-- Stripe Elements will be inserted here -->
                        </div>
                        <div id="card-errors" class="mt-2 text-sm text-red-600 hidden"></div>
                    </div>
                </div>

                <!-- PayPal Payment Form -->
                <div id="paypal-form" class="mb-6 hidden">
                    <div class="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                        <h4 class="text-sm font-medium text-gray-900 dark:text-white mb-3">PayPal Payment</h4>
                        <p class="text-sm text-gray-600 dark:text-gray-400 mb-3">You will be redirected to PayPal to complete your payment.</p>
                        <div id="paypal-button-container"></div>
                    </div>
                </div>

                <!-- Submit Button -->
                <div class="flex items-center justify-between">
                    <a href="{{ route('wallet.index') }}" class="text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200">
                        Cancel
                    </a>
                    <button type="button" id="deposit-btn" 
                            class="bg-blue-600 hover:bg-blue-700 text-white font-medium py-3 px-6 rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed">
                        Deposit Funds
                    </button>
                </div>
            </div>
        </div>

        <!-- Security Notice -->
        <div class="mt-8 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <svg class="h-5 w-5 text-blue-400" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"></path>
                    </svg>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-blue-800 dark:text-blue-200">Secure Payment</h3>
                    <div class="mt-2 text-sm text-blue-700 dark:text-blue-300">
                        <p>Your payment information is encrypted and secure. We use industry-standard SSL encryption to protect your data.</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://js.stripe.com/v3/"></script>
<script src="https://www.paypal.com/sdk/js?client-id={{ $stripeKey }}&currency=USD"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const stripe = Stripe('{{ $stripeKey }}');
    const elements = stripe.elements();
    const cardElement = elements.create('card');
    cardElement.mount('#card-element');

    // Payment method selection
    const paymentMethods = document.querySelectorAll('input[name="payment_method"]');
    const stripeForm = document.getElementById('stripe-form');
    const paypalForm = document.getElementById('paypal-form');

    paymentMethods.forEach(method => {
        method.addEventListener('change', function() {
            if (this.value === 'stripe') {
                stripeForm.classList.remove('hidden');
                paypalForm.classList.add('hidden');
            } else {
                stripeForm.classList.add('hidden');
                paypalForm.classList.remove('hidden');
            }
        });
    });

    // Deposit button click
    document.getElementById('deposit-btn').addEventListener('click', function() {
        const amount = document.getElementById('amount').value;
        const paymentMethod = document.querySelector('input[name="payment_method"]:checked').value;

        if (!amount || amount < 1) {
            alert('Please enter a valid amount');
            return;
        }

        if (paymentMethod === 'stripe') {
            processStripePayment(amount);
        } else if (paymentMethod === 'paypal') {
            processPayPalPayment(amount);
        }
    });

    function processStripePayment(amount) {
        const button = document.getElementById('deposit-btn');
        button.disabled = true;
        button.textContent = 'Processing...';

        stripe.confirmCardPayment('{{ csrf_token() }}', {
            payment_method: {
                card: cardElement,
                billing_details: {
                    name: '{{ $user->name }}',
                    email: '{{ $user->email }}'
                }
            },
            amount: Math.round(amount * 100)
        }).then(function(result) {
            if (result.error) {
                showError(result.error.message);
                button.disabled = false;
                button.textContent = 'Deposit Funds';
            } else {
                // Payment successful
                window.location.href = '{{ route("stripe.success") }}';
            }
        });
    }

    function processPayPalPayment(amount) {
        // Redirect to PayPal or show PayPal button
        alert('PayPal integration will be implemented here');
    }

    function showError(message) {
        const errorDiv = document.getElementById('card-errors');
        errorDiv.textContent = message;
        errorDiv.classList.remove('hidden');
    }
});
</script>
@endpush
@endsection
