@extends('front_end.layouts.auth')

@section('title', 'Register - Socialink')

@if (setting('recaptcha') == 1)
    @push('styles')
        <script src="https://www.google.com/recaptcha/api.js" async defer></script>
    @endpush
@endif

@push('styles')
    @if (setting('password_complexity') == 1)
        <style>
            .password-strength-meter {
                margin-top: 8px;
                height: 4px;
                border-radius: 2px;
                background-color: #e9ecef;
                overflow: hidden;
            }

            .password-strength-meter .strength-bar {
                height: 100%;
                transition: all 0.3s ease;
                width: 0%;
            }

            .password-strength-meter .strength-bar.weak {
                background-color: #dc3545;
                width: 25%;
            }

            .password-strength-meter .strength-bar.medium {
                background-color: #ffc107;
                width: 50%;
            }

            .password-strength-meter .strength-bar.strong {
                background-color: #28a745;
                width: 75%;
            }

            .password-strength-meter .strength-bar.very-strong {
                background-color: #20c997;
                width: 100%;
            }

            .password-requirements {
                margin-top: 8px;
                font-size: 0.875rem;
                color: #6c757d;
            }

            .password-requirements .requirement {
                display: flex;
                align-items: center;
                margin-bottom: 4px;
            }

            .password-requirements .requirement i {
                margin-right: 6px;
                font-size: 12px;
            }

            .password-requirements .requirement.met {
                color: #28a745;
            }

            .password-requirements .requirement.not-met {
                color: #6c757d;
            }

            .password-strength-text {
                font-size: 0.875rem;
                margin-top: 4px;
                font-weight: 500;
            }

            .password-strength-text.weak {
                color: #dc3545;
            }

            .password-strength-text.medium {
                color: #ffc107;
            }

            .password-strength-text.strong {
                color: #28a745;
            }

            .password-strength-text.very-strong {
                color: #20c997;
            }
        </style>
    @endif
@endpush

@push('scripts')
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.querySelector('form[action="{{ route('register') }}"]');
            if (form) {
                form.addEventListener('submit', function(e) {
                    @if (setting('recaptcha') == 1)
                        const recaptchaResponse = grecaptcha.getResponse();
                        if (!recaptchaResponse) {
                            e.preventDefault();
                            alert('Please complete the reCAPTCHA verification before submitting the form.');
                            return false;
                        }
                    @endif

                    @if (setting('password_complexity') == 1)
                        const password = document.getElementById('password').value;
                        const passwordStrength = checkPasswordStrength(password);

                        if (passwordStrength.score < 2) {
                            e.preventDefault();
                            alert(
                                'Please choose a stronger password. Your password should be at least medium strength.'
                                );
                            return false;
                        }
                    @endif

                    // Check if terms checkbox is checked
                    const termsCheckbox = document.getElementById('terms');
                    if (!termsCheckbox.checked) {
                        e.preventDefault();
                        alert('Please agree to the Terms & Conditions to continue.');
                        termsCheckbox.focus();
                        return false;
                    }
                });
            }

            @if (setting('password_complexity') == 1)
                // Password complexity system
                const passwordInput = document.getElementById('password');
                const confirmPasswordInput = document.getElementById('password_confirmation');

                if (passwordInput) {
                    passwordInput.addEventListener('input', function() {
                        checkPasswordComplexity(this.value);
                    });
                }

                function checkPasswordComplexity(password) {
                    const requirements = {
                        length: password.length >= 8,
                        uppercase: /[A-Z]/.test(password),
                        lowercase: /[a-z]/.test(password),
                        numbers: /\d/.test(password),
                        special: /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)
                    };

                    // Update requirement indicators
                    Object.keys(requirements).forEach(req => {
                        const element = document.querySelector(`.requirement-${req}`);
                        if (element) {
                            const icon = element.querySelector('i');
                            if (requirements[req]) {
                                element.classList.add('met');
                                element.classList.remove('not-met');
                                icon.className = 'bi bi-check-circle-fill';
                            } else {
                                element.classList.add('not-met');
                                element.classList.remove('met');
                                icon.className = 'bi bi-circle';
                            }
                        }
                    });

                    // Calculate strength score
                    const score = Object.values(requirements).filter(Boolean).length;
                    const strengthBar = document.querySelector('.strength-bar');
                    const strengthText = document.querySelector('.password-strength-text');

                    if (strengthBar && strengthText) {
                        strengthBar.className = 'strength-bar';

                        if (score <= 2) {
                            strengthBar.classList.add('weak');
                            strengthText.textContent = 'Weak';
                            strengthText.className = 'password-strength-text weak';
                        } else if (score === 3) {
                            strengthBar.classList.add('medium');
                            strengthText.textContent = 'Medium';
                            strengthText.className = 'password-strength-text medium';
                        } else if (score === 4) {
                            strengthBar.classList.add('strong');
                            strengthText.textContent = 'Strong';
                            strengthText.className = 'password-strength-text strong';
                        } else {
                            strengthBar.classList.add('very-strong');
                            strengthText.textContent = 'Very Strong';
                            strengthText.className = 'password-strength-text very-strong';
                        }
                    }

                    return {
                        requirements,
                        score
                    };
                }

                function checkPasswordStrength(password) {
                    const requirements = {
                        length: password.length >= 8,
                        uppercase: /[A-Z]/.test(password),
                        lowercase: /[a-z]/.test(password),
                        numbers: /\d/.test(password),
                        special: /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)
                    };

                    const score = Object.values(requirements).filter(Boolean).length;

                    return {
                        requirements,
                        score
                    };
                }
            @endif
        });
    </script>
@endpush

@section('content')
    <div class="auth-page d-flex align-items-center py-5">
        <div class="auth-inner container">
            <div class="row justify-content-between align-items-center">
                <div class="col-lg-6 col-xl-7 mb-5 mb-lg-0">
                    <div class="brand-panel">
                        <img src="{{ asset('assets/images/logo.png') }}" alt="Socialink" class="brand-mark mb-3">
                        <h1 class="brand-heading">Create your Socialink identity and keep your circles close from day one.
                        </h1>
                        <p class="brand-tagline">Set up a profile that stays consistent across stories, events, games, and
                            marketplace.</p>

                        <div class="brand-highlight">
                            <div class="brand-highlight-icon">
                                <i class="fas fa-handshake-angle"></i>
                            </div>
                            <div class="brand-highlight-text">
                                <strong>Launch your profile in minutes.</strong>
                                <span>Customize privacy, add your story, and sync contacts without the hassle.</span>
                            </div>
                        </div>

                        <div class="stats-glance">
                            <div class="glance-card">
                                <span class="label">Profiles completed</span>
                                <span class="value">92%</span>
                                <span class="subtext">of new members finish setup within 3 minutes</span>
                            </div>
                            <div class="glance-card">
                                <span class="label">Communities to join</span>
                                <span class="value">8.3K</span>
                                <span class="subtext">interest hubs waiting for your first post</span>
                            </div>
                        </div>

                        <div class="trust-badges">
                            <div class="trust-avatars">
                                <span class="trust-avatar variant-blue">CX</span>
                                <span class="trust-avatar variant-green">LM</span>
                                <span class="trust-avatar variant-purple">RA</span>
                                <span class="trust-avatar variant-orange">TN</span>
                            </div>
                            <div class="trust-meta">Creators and teams from <strong>190+ regions</strong> grow together
                                here.</div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-5 col-xl-4">
                    <div class="auth-card card border-0 shadow-sm">
                        <div class="card-body">
                            <h2 class="card-title text-center mb-2">Create a new account</h2>
                            <p class="auth-subtitle text-center mb-4">It's quick, secure, and takes less than a minute.</p>

                            @if ($errors->any())
                                <div class="alert alert-danger">
                                    <ul class="mb-0">
                                        @foreach ($errors->all() as $error)
                                            <li>{{ $error }}</li>
                                        @endforeach
                                    </ul>
                                </div>
                            @endif

                            @if (setting('user_registration') == 1)
                                <form action="{{ route('register') }}" method="POST" novalidate>
                                    @csrf

                                    <div class="row g-3">
                                        <div class="col-12 col-sm-6">
                                            <input type="text" class="form-control" name="first_name"
                                                placeholder="First name" value="{{ old('first_name') }}" required>
                                        </div>
                                        <div class="col-12 col-sm-6">
                                            <input type="text" class="form-control" name="last_name"
                                                placeholder="Surname" value="{{ old('last_name') }}" required>
                                        </div>
                                    </div>

                                    @if (setting('auto_username') == 0)
                                        <div class="mt-3">
                                            <input type="text" class="form-control" name="username"
                                                placeholder="Username" value="{{ old('username') }}" required>
                                        </div>
                                    @endif

                                    <div class="mt-3">
                                        <input type="email" class="form-control" name="email"
                                            placeholder="Email address" value="{{ old('email') }}" required>
                                    </div>

                                    <div class="mt-3">
                                        <div class="position-relative">
                                            <input type="password" class="form-control" id="password" name="password"
                                                placeholder="New password" required>
                                            <i class="fas fa-eye-slash password-toggle" data-target="password"
                                                role="button" aria-label="Toggle password visibility"></i>
                                        </div>

                                        @if (setting('password_complexity') == 1)
                                            <div class="password-strength-meter mt-2">
                                                <div class="strength-bar"></div>
                                            </div>
                                            <div class="password-strength-text"></div>
                                            <div class="password-requirements mt-2">
                                                <div class="requirement requirement-length not-met">
                                                    <i class="bi bi-circle"></i>
                                                    <span>At least 8 characters</span>
                                                </div>
                                                <div class="requirement requirement-uppercase not-met">
                                                    <i class="bi bi-circle"></i>
                                                    <span>At least one uppercase letter</span>
                                                </div>
                                                <div class="requirement requirement-lowercase not-met">
                                                    <i class="bi bi-circle"></i>
                                                    <span>At least one lowercase letter</span>
                                                </div>
                                                <div class="requirement requirement-numbers not-met">
                                                    <i class="bi bi-circle"></i>
                                                    <span>At least one number</span>
                                                </div>
                                                <div class="requirement requirement-special not-met">
                                                    <i class="bi bi-circle"></i>
                                                    <span>At least one special character</span>
                                                </div>
                                            </div>
                                        @endif
                                    </div>

                                    <div class="mt-3 position-relative">
                                        <input type="password" class="form-control" id="password_confirmation"
                                            name="password_confirmation" placeholder="Confirm password" required>
                                        <i class="fas fa-eye-slash password-toggle" data-target="password_confirmation"
                                            role="button" aria-label="Toggle password visibility"></i>
                                    </div>

                                    <div class="mt-3">
                                        <label class="form-label text-secondary mb-1">Birthday</label>
                                        <input type="date" class="form-control" name="date_of_birth"
                                            value="{{ old('date_of_birth') }}" required>
                                    </div>

                                    <div class="mt-3">
                                        <label class="form-label text-secondary mb-1">Gender</label>
                                        <select class="form-control" name="gender" required>
                                            <option value="" disabled {{ old('gender') ? '' : 'selected' }}>Select
                                                gender</option>
                                            <option value="Male" {{ old('gender') == 'Male' ? 'selected' : '' }}>Male
                                            </option>
                                            <option value="Female" {{ old('gender') == 'Female' ? 'selected' : '' }}>
                                                Female</option>
                                        </select>
                                    </div>

                                    @if (setting('recaptcha') == 1)
                                        <div class="mt-3">
                                            <div class="g-recaptcha" data-sitekey="{{ setting('recaptcha_key') }}"></div>
                                            @if ($errors->has('recaptcha'))
                                                <div class="text-danger mt-2">
                                                    <small>{{ $errors->first('recaptcha') }}</small>
                                                </div>
                                            @endif
                                        </div>
                                    @endif

                                    <div class="form-check mt-3 {{ $errors->has('terms') ? 'is-invalid' : '' }}">
                                        <input type="checkbox" class="form-check-input {{ $errors->has('terms') ? 'is-invalid' : '' }}" 
                                               id="terms" name="terms" value="1" required {{ old('terms') ? 'checked' : '' }}>
                                        <label class="form-check-label" for="terms">
                                            I agree to the <a href="{{ url('page/terms-and-conditions') }}"
                                                class="link-blue text-decoration-none">Terms &amp; Conditions</a>.
                                        </label>
                                        @if ($errors->has('terms'))
                                            <div class="invalid-feedback d-block">
                                                {{ $errors->first('terms') }}
                                            </div>
                                        @endif
                                    </div>

                                    <button type="submit" class="btn btn-primary w-100">Sign Up</button>
                                </form>
                            @else
                                <p class="text-danger text-center mb-0">User registration is currently disabled.</p>
                            @endif

                            <p class="text-center text-secondary mt-4 mb-0">
                                Already have an account?
                                <a href="{{ route('login') }}" class="link-blue text-decoration-none">Log in</a>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection
