@extends('admin.layouts.home')

@section('title', 'Blacklist Management')

@section('content')
<div class="container-fluid">
    <div>
        <h3>Blacklist Users</h3>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item">
                    <a href="{{ url('admin') }}">Home</a>
                </li>
                <li class="breadcrumb-item">
                    <a href="{{ route('admin.tools.index') }}">Tools</a>
                </li>
                <li class="breadcrumb-item active" aria-current="page">Blacklist Users</li>
            </ol>
        </nav>
    </div>
    
    <!-- Add to Blacklist Form -->
    <div class="row">
        <div class="col-lg-12 col-md-12">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">Ban IP, Email or Username</h5>
                    
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> Add any email, username, IP address, or email domain to prevent users from creating accounts or accessing the site.
                    </div>
                    
                    @if(session('success'))
                        <div class="alert alert-success">
                            {{ session('success') }}
                        </div>
                    @endif
                    
                    @if(session('error'))
                        <div class="alert alert-danger">
                            {{ session('error') }}
                        </div>
                    @endif
                    
                    <form action="{{ route('admin.tools.blacklist.add') }}" method="POST">
                        @csrf
                        
                        <div class="row mb-3">
                            <div class="col-md-8">
                                <div class="form-group">
                                    <label for="value" class="form-label">IP Address, Email Address, Email Domain or Username</label>
                                    <input type="text" id="value" name="value" class="form-control" required>
                                    <div class="form-text">Examples: 192.168.1.1, user@example.com, username, example.com</div>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="type" class="form-label">Type</label>
                                    <select class="form-select" id="type" name="type" required>
                                        <option value="ip">IP Address</option>
                                        <option value="email">Email Address</option>
                                        <option value="username">Username</option>
                                        <option value="email_domain">Email Domain</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-ban"></i> Ban
                        </button>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Manage Blacklist -->
        <div class="col-lg-12 col-md-12 mt-4">
            <div class="card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-4">
                        <h5 class="card-title">Manage Blacklisted Items</h5>
                        
                        <form action="{{ route('admin.tools.blacklist.search') }}" method="GET" class="d-flex">
                            <input type="text" name="query" class="form-control me-2" placeholder="Search..." value="{{ $query ?? '' }}">
                            <button type="submit" class="btn btn-info">
                                <i class="fas fa-search"></i>
                            </button>
                        </form>
                    </div>
                    
                    @if(count($blacklisted) > 0)
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th width="5%">
                                            <div class="form-check">
                                                <input class="form-check-input check-all" type="checkbox" id="check-all">
                                                <label class="form-check-label" for="check-all"></label>
                                            </div>
                                        </th>
                                        <th width="5%">ID</th>
                                        <th width="15%">Type</th>
                                        <th width="40%">Value</th>
                                        <th width="20%">Added On</th>
                                        <th width="15%">Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($blacklisted as $item)
                                        <tr class="blacklist-item" id="item-{{ $item->id }}" data-id="{{ $item->id }}">
                                            <td>
                                                <div class="form-check">
                                                    <input class="form-check-input delete-checkbox" type="checkbox" id="check-{{ $item->id }}" data-id="{{ $item->id }}">
                                                    <label class="form-check-label" for="check-{{ $item->id }}"></label>
                                                </div>
                                            </td>
                                            <td>{{ $item->id }}</td>
                                            <td>
                                                @if($item->type == 'ip')
                                                    <span class="badge bg-danger">IP Address</span>
                                                @elseif($item->type == 'email')
                                                    <span class="badge bg-warning">Email</span>
                                                @elseif($item->type == 'username')
                                                    <span class="badge bg-info">Username</span>
                                                @elseif($item->type == 'email_domain')
                                                    <span class="badge bg-primary">Email Domain</span>
                                                @endif
                                            </td>
                                            <td>{{ $item->value }}</td>
                                            <td>{{ \Carbon\Carbon::parse($item->created_at)->format('M d, Y h:i A') }}</td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-danger" onclick="confirmDelete({{ $item->id }})">
                                                    <i class="fas fa-trash-alt"></i> Delete
                                                </button>
                                                
                                                <form id="delete-form-{{ $item->id }}" action="{{ route('admin.tools.blacklist.remove', $item->id) }}" method="POST" style="display: none;">
                                                    @csrf
                                                    @method('DELETE')
                                                </form>
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="d-flex justify-content-between align-items-center mt-4">
                            <div>
                                Showing {{ $blacklisted->firstItem() ?? 0 }} to {{ $blacklisted->lastItem() ?? 0 }} of {{ $blacklisted->total() }} entries
                            </div>
                            
                            <div>
                                {{ $blacklisted->links('pagination::bootstrap-5') }}
                            </div>
                        </div>
                        
                        <div class="mt-3">
                            <button type="button" class="btn btn-danger delete-selected" disabled>
                                <i class="fas fa-trash-alt"></i> Delete Selected <span class="selected-count"></span>
                            </button>
                        </div>
                    @else
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> No blacklisted items found.
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteModalLabel">Confirm Deletion</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to remove this item from the blacklist?
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteBtn">Delete</button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Multiple Modal -->
<div class="modal fade" id="deleteMultipleModal" tabindex="-1" aria-labelledby="deleteMultipleModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteMultipleModalLabel">Confirm Multiple Deletion</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to remove the selected items from the blacklist?
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteMultipleBtn">Delete</button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Handle single item deletion
    function confirmDelete(id) {
        const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
        const confirmBtn = document.getElementById('confirmDeleteBtn');
        
        confirmBtn.onclick = function() {
            document.getElementById('delete-form-' + id).submit();
        };
        
        modal.show();
    }
    
    // Handle check all functionality
    document.getElementById('check-all').addEventListener('change', function() {
        const isChecked = this.checked;
        document.querySelectorAll('.delete-checkbox').forEach(function(checkbox) {
            checkbox.checked = isChecked;
        });
        updateDeleteSelectedButton();
    });
    
    // Handle individual checkbox changes
    document.querySelectorAll('.delete-checkbox').forEach(function(checkbox) {
        checkbox.addEventListener('change', function() {
            updateDeleteSelectedButton();
        });
    });
    
    // Update delete selected button state
    function updateDeleteSelectedButton() {
        const selectedCount = document.querySelectorAll('.delete-checkbox:checked').length;
        const deleteSelectedBtn = document.querySelector('.delete-selected');
        const selectedCountSpan = document.querySelector('.selected-count');
        
        if (selectedCount > 0) {
            deleteSelectedBtn.disabled = false;
            selectedCountSpan.textContent = '(' + selectedCount + ')';
        } else {
            deleteSelectedBtn.disabled = true;
            selectedCountSpan.textContent = '';
        }
    }
    
    // Handle delete selected button click
    document.querySelector('.delete-selected').addEventListener('click', function() {
        const modal = new bootstrap.Modal(document.getElementById('deleteMultipleModal'));
        modal.show();
    });
    
    // Handle confirm delete multiple
    document.getElementById('confirmDeleteMultipleBtn').addEventListener('click', function() {
        const selectedIds = [];
        document.querySelectorAll('.delete-checkbox:checked').forEach(function(checkbox) {
            selectedIds.push(checkbox.getAttribute('data-id'));
        });
        
        if (selectedIds.length > 0) {
            // Send AJAX request to delete multiple items
            fetch('{{ route("admin.tools.blacklist.remove-multiple") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ ids: selectedIds })
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    // Remove deleted items from the DOM
                    selectedIds.forEach(function(id) {
                        document.getElementById('item-' + id).remove();
                    });
                    
                    // Close modal and update button state
                    bootstrap.Modal.getInstance(document.getElementById('deleteMultipleModal')).hide();
                    updateDeleteSelectedButton();
                    
                    // Show success message
                    const alertDiv = document.createElement('div');
                    alertDiv.className = 'alert alert-success';
                    alertDiv.innerHTML = '<i class="fas fa-check-circle"></i> Selected items have been removed from the blacklist.';
                    
                    const cardBody = document.querySelector('.card-body');
                    cardBody.insertBefore(alertDiv, cardBody.firstChild);
                    
                    // Auto-hide the alert after 3 seconds
                    setTimeout(function() {
                        alertDiv.remove();
                    }, 3000);
                }
            })
            .catch(error => {
                console.error('Error:', error);
            });
        }
    });
</script>
@endpush 