@extends('admin.layouts.home')

@section('title', 'Edit Movie')

@section('content')
<div class="container-fluid">
    <!-- Page Heading -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3 mb-0 text-gray-800">Edit Movie: {{ $movie->title }}</h1>
        <div>
            <a href="{{ route('movies.show', $movie->slug) }}" class="btn btn-info mr-2" target="_blank">
                <i class="fas fa-eye mr-1"></i> View Movie
            </a>
            <a href="{{ route('admin.movies.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left mr-1"></i> Back to Movies
            </a>
        </div>
    </div>

    <!-- Movie Form Card -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Edit Movie Information</h6>
        </div>
        <div class="card-body">
            @if($errors->any())
            <div class="alert alert-danger">
                <ul class="mb-0">
                    @foreach($errors->all() as $error)
                    <li>{{ $error }}</li>
                    @endforeach
                </ul>
            </div>
            @endif

            <form action="{{ route('admin.movies.update', $movie) }}" method="POST" enctype="multipart/form-data">
                @csrf
                @method('PUT')
                <div class="row">
                    <div class="col-md-8">
                        <!-- Basic Information -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Basic Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="form-group">
                                    <label for="title">Title <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="title" name="title" value="{{ old('title', $movie->title) }}" required>
                                </div>
                                <div class="form-group">
                                    <label for="description">Description <span class="text-danger">*</span></label>
                                    <textarea class="form-control" id="description" name="description" rows="5" required>{{ old('description', $movie->description) }}</textarea>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="genre_id">Genre <span class="text-danger">*</span></label>
                                            <select class="form-control" id="genre_id" name="genre_id" required>
                                                <option value="">Select Genre</option>
                                                @foreach($genres as $genre)
                                                <option value="{{ $genre->id }}" {{ old('genre_id', $movie->genre_id) == $genre->id ? 'selected' : '' }}>
                                                    {{ $genre->name }}
                                                </option>
                                                @endforeach
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="content_rating">Content Rating <span class="text-danger">*</span></label>
                                            <select class="form-control" id="content_rating" name="content_rating" required>
                                                <option value="">Select Rating</option>
                                                <option value="G" {{ old('content_rating', $movie->content_rating) == 'G' ? 'selected' : '' }}>G</option>
                                                <option value="PG" {{ old('content_rating', $movie->content_rating) == 'PG' ? 'selected' : '' }}>PG</option>
                                                <option value="PG-13" {{ old('content_rating', $movie->content_rating) == 'PG-13' ? 'selected' : '' }}>PG-13</option>
                                                <option value="R" {{ old('content_rating', $movie->content_rating) == 'R' ? 'selected' : '' }}>R</option>
                                                <option value="NC-17" {{ old('content_rating', $movie->content_rating) == 'NC-17' ? 'selected' : '' }}>NC-17</option>
                                                <option value="Not Rated" {{ old('content_rating', $movie->content_rating) == 'Not Rated' ? 'selected' : '' }}>Not Rated</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="release_year">Release Year <span class="text-danger">*</span></label>
                                            <input type="number" class="form-control" id="release_year" name="release_year" min="1900" max="{{ date('Y') }}" value="{{ old('release_year', $movie->release_year) }}" required>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="duration">Duration (minutes) <span class="text-danger">*</span></label>
                                            <input type="number" class="form-control" id="duration" name="duration" min="1" value="{{ old('duration', $movie->duration) }}" required>
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="director">Director</label>
                                            <input type="text" class="form-control" id="director" name="director" value="{{ old('director', $movie->director) }}">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="cast">Cast (comma separated)</label>
                                            <input type="text" class="form-control" id="cast" name="cast" value="{{ old('cast', is_array($movie->cast) ? implode(', ', $movie->cast) : $movie->cast) }}" placeholder="Actor 1, Actor 2, Actor 3">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Media Information -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Media Information</h6>
                            </div>
                            <div class="card-body">
                                <div class="form-group">
                                    <label for="video_url">Video URL <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="video_url" name="video_url" value="{{ old('video_url', $movie->video_url) }}" required>
                                    <small class="form-text text-muted">Enter the embed URL for the movie (e.g., YouTube or Vimeo embed URL)</small>
                                </div>
                                <div class="form-group">
                                    <label for="trailer_url">Trailer URL</label>
                                    <input type="text" class="form-control" id="trailer_url" name="trailer_url" value="{{ old('trailer_url', $movie->trailer_url) }}">
                                    <small class="form-text text-muted">Enter the embed URL for the trailer (e.g., YouTube or Vimeo embed URL)</small>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <!-- Poster Upload -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Movie Poster</h6>
                            </div>
                            <div class="card-body">
                                <div class="form-group">
                                    <label for="poster">Upload New Poster</label>
                                    <input type="file" class="form-control-file" id="poster" name="poster" accept="image/jpeg,image/png,image/jpg">
                                    <small class="form-text text-muted">Recommended size: 300x450 pixels. Max file size: 2MB.</small>
                                </div>
                                <div class="mt-3 text-center">
                                    <div class="img-thumbnail p-2" style="height: 220px; width: 160px; margin: 0 auto;">
                                        @if($movie->poster_url)
                                        <img src="{{ $movie->poster_url }}" class="h-100" style="object-fit: contain;" alt="{{ $movie->title }}">
                                        @else
                                        <div class="d-flex align-items-center justify-content-center h-100 bg-light">
                                            <i class="fas fa-film fa-3x text-secondary"></i>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Statistics -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Statistics</h6>
                            </div>
                            <div class="card-body">
                                <p><strong>Views:</strong> {{ number_format($movie->views_count) }}</p>
                                <p><strong>Average Rating:</strong> {{ number_format($movie->avgRating, 1) }}/5</p>
                                <p><strong>Total Ratings:</strong> {{ $movie->ratings->count() }}</p>
                                <p><strong>Added On:</strong> {{ $movie->created_at->format('M d, Y') }}</p>
                                <p><strong>Last Updated:</strong> {{ $movie->updated_at->format('M d, Y') }}</p>
                            </div>
                        </div>

                        <!-- Settings -->
                        <div class="card mb-4">
                            <div class="card-header">
                                <h6 class="mb-0">Settings</h6>
                            </div>
                            <div class="card-body">
                                <div class="form-group">
                                    <div class="custom-control custom-switch">
                                        <input type="checkbox" class="custom-control-input" id="is_active" name="is_active" value="1" {{ old('is_active', $movie->is_active) ? 'checked' : '' }}>
                                        <label class="custom-control-label" for="is_active">Active</label>
                                    </div>
                                    <small class="form-text text-muted">If enabled, this movie will be visible to users.</small>
                                </div>
                                <div class="form-group">
                                    <div class="custom-control custom-switch">
                                        <input type="checkbox" class="custom-control-input" id="featured" name="featured" value="1" {{ old('featured', $movie->featured) ? 'checked' : '' }}>
                                        <label class="custom-control-label" for="featured">Featured</label>
                                    </div>
                                    <small class="form-text text-muted">If enabled, this movie will be highlighted in featured sections.</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="text-center mt-4">
                    <button type="submit" class="btn btn-primary btn-lg px-5">
                        <i class="fas fa-save mr-2"></i> Update Movie
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
    // Preview uploaded image
    document.getElementById('poster').onchange = function(e) {
        const preview = document.querySelector('.img-thumbnail');
        const file = e.target.files[0];
        
        if (file) {
            const reader = new FileReader();
            
            reader.onload = function(e) {
                preview.innerHTML = `<img src="${e.target.result}" class="h-100" style="object-fit: contain;">`;
            }
            
            reader.readAsDataURL(file);
        }
    };
</script>
@endpush