@extends('admin.layouts.home')

@section('title', 'Order Details')

@section('styles')
<style>
    .detail-card {
        border-radius: 10px;
        overflow: hidden;
        transition: transform 0.3s ease, box-shadow 0.3s ease;
    }
    
    .detail-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 10px 20px rgba(0, 0, 0, 0.1);
    }
    
    .status-badge {
        padding: 5px 10px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 500;
    }
    
    .timeline {
        position: relative;
        padding-left: 30px;
    }
    
    .timeline::before {
        content: '';
        position: absolute;
        left: 10px;
        top: 0;
        height: 100%;
        width: 2px;
        background-color: #dee2e6;
    }
    
    .timeline-item {
        position: relative;
        padding-bottom: 20px;
    }
    
    .timeline-item:last-child {
        padding-bottom: 0;
    }
    
    .timeline-marker {
        position: absolute;
        left: -30px;
        top: 0;
        width: 20px;
        height: 20px;
        border-radius: 50%;
        background-color: #fff;
        border: 2px solid #dee2e6;
        z-index: 1;
    }
    
    .timeline-marker.active {
        background-color: #0d6efd;
        border-color: #0d6efd;
    }
    
    .timeline-marker.completed {
        background-color: #198754;
        border-color: #198754;
    }
    
    .timeline-marker.cancelled {
        background-color: #dc3545;
        border-color: #dc3545;
    }
    
    .product-image {
        width: 80px;
        height: 80px;
        object-fit: cover;
        border-radius: 8px;
    }
</style>
@endsection

@section('content')
<div class="container-fluid px-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mt-4">Order Details</h1>
        <div>
            <a href="{{ route('admin.marketplace.orders') }}" class="btn btn-outline-primary">
                <i class="fas fa-arrow-left me-1"></i> Back to Orders
            </a>
        </div>
    </div>
    
    <!-- Order Header -->
    <div class="card detail-card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <h2 class="mb-3">Order #{{ $order->order_number }}</h2>
                    <div class="d-flex align-items-center mb-3">
                        @php
                            $statusClass = [
                                'pending' => 'bg-warning text-dark',
                                'processing' => 'bg-info text-white',
                                'shipped' => 'bg-primary',
                                'delivered' => 'bg-success',
                                'completed' => 'bg-success',
                                'cancelled' => 'bg-danger',
                            ][$order->status] ?? 'bg-secondary';
                        @endphp
                        <span class="status-badge {{ $statusClass }} me-2">
                            {{ ucfirst($order->status) }}
                        </span>
                        <span class="text-muted">{{ $order->created_at->format('M d, Y H:i') }}</span>
                    </div>
                    <p class="mb-0">
                        <strong>Payment Method:</strong> {{ ucfirst($order->payment_method ?? 'N/A') }}
                    </p>
                    <p>
                        <strong>Payment Status:</strong> 
                        @if($order->payment_status == 'paid')
                            <span class="badge bg-success">Paid</span>
                        @elseif($order->payment_status == 'pending')
                            <span class="badge bg-warning text-dark">Pending</span>
                        @elseif($order->payment_status == 'failed')
                            <span class="badge bg-danger">Failed</span>
                        @else
                            <span class="badge bg-secondary">{{ ucfirst($order->payment_status ?? 'N/A') }}</span>
                        @endif
                    </p>
                </div>
                <div class="col-md-6 text-md-end">
                    <button type="button" class="btn btn-warning mb-3" data-bs-toggle="modal" data-bs-target="#statusModal">
                        <i class="fas fa-sync-alt me-1"></i> Update Status
                    </button>
                    <p class="mb-0">
                        <strong>Total Amount:</strong> <span class="fs-4 text-primary">${{ number_format($order->total_amount, 2) }}</span>
                    </p>
                    @if($order->discount_amount > 0)
                        <p class="mb-0 text-success">
                            <strong>Discount Applied:</strong> ${{ number_format($order->discount_amount, 2) }}
                        </p>
                    @endif
                </div>
            </div>
        </div>
    </div>
    
    <div class="row">
        <!-- Customer & Shipping Info -->
        <div class="col-lg-4 mb-4">
            <div class="card detail-card h-100">
                <div class="card-header">
                    <i class="fas fa-user me-1"></i>
                    Customer Information
                </div>
                <div class="card-body">
                    @if($order->user)
                        <div class="d-flex align-items-center mb-3">
                            @if($order->user->avatar)
                                <img src="{{ asset('storage/' . $order->user->avatar) }}" alt="{{ $order->user->name }}" class="rounded-circle me-3" style="width: 50px; height: 50px; object-fit: cover;">
                            @else
                                <div class="rounded-circle bg-light d-flex align-items-center justify-content-center me-3" style="width: 50px; height: 50px;">
                                    <i class="fas fa-user text-secondary"></i>
                                </div>
                            @endif
                            <div>
                                <h5 class="mb-0">{{ $order->user->name }}</h5>
                                <p class="mb-0 text-muted">{{ $order->user->email }}</p>
                            </div>
                        </div>
                    @else
                        <div class="alert alert-info">
                            Guest Order
                        </div>
                    @endif
                    
                    <h6 class="mt-4 mb-3">Shipping Address</h6>
                    @if($order->address)
                        <address>
                            <strong>{{ $order->address->full_name }}</strong><br>
                            {{ $order->address->address_line1 }}<br>
                            @if($order->address->address_line2)
                                {{ $order->address->address_line2 }}<br>
                            @endif
                            {{ $order->address->city }}, {{ $order->address->state }} {{ $order->address->postal_code }}<br>
                            {{ $order->address->country }}<br>
                            <abbr title="Phone">P:</abbr> {{ $order->address->phone }}
                        </address>
                    @else
                        <p class="text-muted">No shipping address provided</p>
                    @endif
                    
                    <h6 class="mt-4 mb-3">Billing Address</h6>
                    @if($order->billing_address)
                        <address>
                            <strong>{{ $order->billing_address->full_name }}</strong><br>
                            {{ $order->billing_address->address_line1 }}<br>
                            @if($order->billing_address->address_line2)
                                {{ $order->billing_address->address_line2 }}<br>
                            @endif
                            {{ $order->billing_address->city }}, {{ $order->billing_address->state }} {{ $order->billing_address->postal_code }}<br>
                            {{ $order->billing_address->country }}<br>
                            <abbr title="Phone">P:</abbr> {{ $order->billing_address->phone }}
                        </address>
                    @elseif($order->address)
                        <p class="text-muted">Same as shipping address</p>
                    @else
                        <p class="text-muted">No billing address provided</p>
                    @endif
                </div>
            </div>
        </div>
        
        <!-- Order Status Timeline -->
        <div class="col-lg-4 mb-4">
            <div class="card detail-card h-100">
                <div class="card-header">
                    <i class="fas fa-history me-1"></i>
                    Order Timeline
                </div>
                <div class="card-body">
                    <div class="timeline">
                        <div class="timeline-item">
                            @php
                                $isPending = in_array($order->status, ['pending', 'processing', 'shipped', 'delivered', 'completed']);
                                $markerClass = $isPending ? 'completed' : '';
                            @endphp
                            <div class="timeline-marker {{ $markerClass }}"></div>
                            <div class="timeline-content">
                                <h6 class="mb-0">Order Placed</h6>
                                <p class="text-muted mb-0 small">{{ $order->created_at->format('M d, Y H:i') }}</p>
                            </div>
                        </div>
                        
                        <div class="timeline-item">
                            @php
                                $isProcessing = in_array($order->status, ['processing', 'shipped', 'delivered', 'completed']);
                                $markerClass = $isProcessing ? 'completed' : ($order->status == 'pending' ? 'active' : '');
                            @endphp
                            <div class="timeline-marker {{ $markerClass }}"></div>
                            <div class="timeline-content">
                                <h6 class="mb-0">Processing</h6>
                                <p class="text-muted mb-0 small">
                                    @if($isProcessing)
                                        {{ $order->updated_at->format('M d, Y H:i') }}
                                    @else
                                        Waiting
                                    @endif
                                </p>
                            </div>
                        </div>
                        
                        <div class="timeline-item">
                            @php
                                $isShipped = in_array($order->status, ['shipped', 'delivered', 'completed']);
                                $markerClass = $isShipped ? 'completed' : ($order->status == 'processing' ? 'active' : '');
                            @endphp
                            <div class="timeline-marker {{ $markerClass }}"></div>
                            <div class="timeline-content">
                                <h6 class="mb-0">Shipped</h6>
                                <p class="text-muted mb-0 small">
                                    @if($isShipped)
                                        {{ $order->updated_at->format('M d, Y H:i') }}
                                    @else
                                        Waiting
                                    @endif
                                </p>
                            </div>
                        </div>
                        
                        <div class="timeline-item">
                            @php
                                $isDelivered = in_array($order->status, ['delivered', 'completed']);
                                $markerClass = $isDelivered ? 'completed' : ($order->status == 'shipped' ? 'active' : '');
                            @endphp
                            <div class="timeline-marker {{ $markerClass }}"></div>
                            <div class="timeline-content">
                                <h6 class="mb-0">Delivered</h6>
                                <p class="text-muted mb-0 small">
                                    @if($isDelivered)
                                        {{ $order->updated_at->format('M d, Y H:i') }}
                                    @else
                                        Waiting
                                    @endif
                                </p>
                            </div>
                        </div>
                        
                        <div class="timeline-item">
                            @php
                                $isCompleted = $order->status == 'completed';
                                $isCancelled = $order->status == 'cancelled';
                                $markerClass = $isCompleted ? 'completed' : ($isCancelled ? 'cancelled' : ($order->status == 'delivered' ? 'active' : ''));
                            @endphp
                            <div class="timeline-marker {{ $markerClass }}"></div>
                            <div class="timeline-content">
                                <h6 class="mb-0">{{ $isCancelled ? 'Cancelled' : 'Completed' }}</h6>
                                <p class="text-muted mb-0 small">
                                    @if($isCompleted || $isCancelled)
                                        {{ $order->updated_at->format('M d, Y H:i') }}
                                    @else
                                        Waiting
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Order Summary -->
        <div class="col-lg-4 mb-4">
            <div class="card detail-card h-100">
                <div class="card-header">
                    <i class="fas fa-receipt me-1"></i>
                    Order Summary
                </div>
                <div class="card-body">
                    <div class="d-flex justify-content-between mb-3">
                        <span>Subtotal:</span>
                        <span>${{ number_format($order->subtotal ?? $order->total_amount, 2) }}</span>
                    </div>
                    
                    @if($order->tax_amount)
                        <div class="d-flex justify-content-between mb-3">
                            <span>Tax:</span>
                            <span>${{ number_format($order->tax_amount, 2) }}</span>
                        </div>
                    @endif
                    
                    @if($order->shipping_amount)
                        <div class="d-flex justify-content-between mb-3">
                            <span>Shipping:</span>
                            <span>${{ number_format($order->shipping_amount, 2) }}</span>
                        </div>
                    @endif
                    
                    @if($order->discount_amount)
                        <div class="d-flex justify-content-between mb-3 text-success">
                            <span>Discount:</span>
                            <span>-${{ number_format($order->discount_amount, 2) }}</span>
                        </div>
                    @endif
                    
                    <hr>
                    
                    <div class="d-flex justify-content-between mb-3 fw-bold">
                        <span>Total:</span>
                        <span>${{ number_format($order->total_amount, 2) }}</span>
                    </div>
                    
                    @if($order->coupon_code)
                        <div class="alert alert-success mt-3 mb-0 py-2">
                            <small>Coupon Applied: <strong>{{ $order->coupon_code }}</strong></small>
                        </div>
                    @endif
                    
                    @if($order->notes)
                        <div class="mt-4">
                            <h6>Order Notes:</h6>
                            <p class="mb-0">{{ $order->notes }}</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
    
    <!-- Order Items -->
    <div class="card detail-card mb-4">
        <div class="card-header">
            <i class="fas fa-box me-1"></i>
            Order Items
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table">
                    <thead>
                        <tr>
                            <th style="width: 80px"></th>
                            <th>Product</th>
                            <th>Store</th>
                            <th class="text-center">Quantity</th>
                            <th class="text-end">Price</th>
                            <th class="text-end">Total</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($order->products as $product)
                            <tr>
                                <td>
                                    @if($product->thumbnail)
                                        <img src="{{ asset('storage/' . $product->thumbnail) }}" alt="{{ $product->name }}" class="product-image">
                                    @else
                                        <div class="product-image bg-light d-flex align-items-center justify-content-center">
                                            <i class="fas fa-box text-secondary"></i>
                                        </div>
                                    @endif
                                </td>
                                <td>
                                    <a href="{{ route('admin.marketplace.products.show', $product->id) }}" class="text-decoration-none">
                                        <div class="fw-bold">{{ $product->name }}</div>
                                        <div class="small text-muted">SKU: {{ $product->sku ?? 'N/A' }}</div>
                                    </a>
                                </td>
                                <td>
                                    @if($product->store)
                                        <a href="{{ route('admin.marketplace.stores.show', $product->store->id) }}" class="text-decoration-none">
                                            {{ $product->store->name }}
                                        </a>
                                    @else
                                        <span class="text-muted">N/A</span>
                                    @endif
                                </td>
                                <td class="text-center">
                                    {{ $product->pivot->quantity ?? 1 }}
                                </td>
                                <td class="text-end">
                                    ${{ number_format($product->pivot->price ?? $product->price, 2) }}
                                </td>
                                <td class="text-end fw-bold">
                                    ${{ number_format(($product->pivot->price ?? $product->price) * ($product->pivot->quantity ?? 1), 2) }}
                                </td>
                            </tr>
                        @endforeach
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan="5" class="text-end">Subtotal:</td>
                            <td class="text-end">${{ number_format($order->subtotal ?? $order->total_amount, 2) }}</td>
                        </tr>
                        @if($order->tax_amount)
                            <tr>
                                <td colspan="5" class="text-end">Tax:</td>
                                <td class="text-end">${{ number_format($order->tax_amount, 2) }}</td>
                            </tr>
                        @endif
                        @if($order->shipping_amount)
                            <tr>
                                <td colspan="5" class="text-end">Shipping:</td>
                                <td class="text-end">${{ number_format($order->shipping_amount, 2) }}</td>
                            </tr>
                        @endif
                        @if($order->discount_amount)
                            <tr class="text-success">
                                <td colspan="5" class="text-end">Discount:</td>
                                <td class="text-end">-${{ number_format($order->discount_amount, 2) }}</td>
                            </tr>
                        @endif
                        <tr>
                            <td colspan="5" class="text-end fw-bold">Total:</td>
                            <td class="text-end fw-bold">${{ number_format($order->total_amount, 2) }}</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Status Modal -->
    <div class="modal fade" id="statusModal" tabindex="-1" aria-labelledby="statusModalLabel" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="statusModalLabel">Update Order Status</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form action="{{ route('admin.marketplace.orders.update-status', $order->id) }}" method="POST">
                    @csrf
                    <div class="modal-body">
                        <p>Update status for Order <strong>#{{ $order->order_number }}</strong>:</p>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="status" id="statusPending" value="pending" {{ $order->status == 'pending' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusPending">
                                Pending
                            </label>
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="status" id="statusProcessing" value="processing" {{ $order->status == 'processing' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusProcessing">
                                Processing
                            </label>
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="status" id="statusShipped" value="shipped" {{ $order->status == 'shipped' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusShipped">
                                Shipped
                            </label>
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="status" id="statusDelivered" value="delivered" {{ $order->status == 'delivered' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusDelivered">
                                Delivered
                            </label>
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="radio" name="status" id="statusCompleted" value="completed" {{ $order->status == 'completed' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusCompleted">
                                Completed
                            </label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="radio" name="status" id="statusCancelled" value="cancelled" {{ $order->status == 'cancelled' ? 'checked' : '' }}>
                            <label class="form-check-label" for="statusCancelled">
                                Cancelled
                            </label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Update Status</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection 