<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Modules\Post\Models\Post;
use Illuminate\Support\Facades\Auth;

class Page extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    // protected $fillable = [
    //     'name',
    //     'description',
    //     'page_category',
    //     'user_id',
    //     'cover_photo',
    //     'website',
    //     'email',
    //     'phone',
    //     'location',
    //     'active',
    //     'featured',
    //     'page_username',
    //     'page_title',
    //     'page_description'
    // ];
    protected  $guarded = [];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'active' => 'boolean',
        'featured' => 'boolean',
        'is_verified' => 'boolean',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the category that the page belongs to.
     */
    public function category()
    {
        return $this->belongsTo(Category::class, 'page_category');
    }

    /**
     * Get the user who created the page.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Get the likes for the page.
     */
    public function likes()
    {
        return $this->hasMany(PageLike::class);
    }

    /**
     * Get the posts for the page.
     */
    public function posts()
    {
        return $this->hasMany(Post::class, 'page_id');
    }

    /**
     * Get the admins for the page.
     */
    public function admins()
    {
        return $this->belongsToMany(User::class, 'page_admins')
            ->withTimestamps();
    }

    /**
     * Get the followers for the page.
     */
    public function followers()
    {
        return $this->belongsToMany(User::class, 'page_followers')
            ->withTimestamps();
    }

    /**
     * Scope a query to only include active pages.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeActive($query)
    {
        return $query->where('active', true);
    }

    /**
     * Scope a query to only include featured pages.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }

    /**
     * Scope a query to only include pages in a specific category.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  int  $categoryId
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeInCategory($query, $categoryId)
    {
        return $query->where('page_category', $categoryId);
    }

    /**
     * Check if a user has liked this page.
     *
     * @param  int  $userId
     * @return bool
     */
    public function isLikedBy($userId)
    {
        return $this->likes()->where('user_id', $userId)->exists();
    }

    /**
     * Check if a user is an admin of this page.
     *
     * @param  int  $userId
     * @return bool
     */
    public function isAdminBy($userId)
    {
        return $this->admins()->where('user_id', $userId)->exists();
    }

    /**
     * Check if a user is following this page.
     *
     * @param  int  $userId
     * @return bool
     */
    public function isFollowedBy($userId)
    {
        return $this->followers()->where('user_id', $userId)->exists();
    }

    /**
     * Check if the page is liked by the authenticated user.
     *
     * @return bool
     */
    public function isLikedByUser($userId = null)
    {
        $userId = $userId ?? Auth::id();
        return $this->likes()->where('user_id', $userId)->exists();
    }
    public function deletePage($page_id)
    {
        PageLike::where('page_id', $page_id)->delete();
        $postModel = new Post();
        $pagePosts = $postModel->where('page_id', $page_id)->pluck('id')->toArray();
        $postModel->deletePosts($pagePosts);
        $this->delete();
    }
}
