<?php

namespace App\Http\Resources;

use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class EventDetailResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    public function toArray($request)
    {
        $userId = Auth::id();
        
        return [
            // Basic event info
            'id' => $this->id,
            'name' => $this->name,
            'slug' => $this->slug,
            'description' => $this->description,
            'location' => $this->location,
            
            // Dates and times
            'dates' => [
                'start_date' => $this->start_date,
                'start_time' => $this->start_time,
                'end_date' => $this->end_date,
                'end_time' => $this->end_time,
                'formatted' => [
                    'start' => Carbon::parse("{$this->start_date} {$this->start_time}")->format('F j, Y \a\t g:i A'),
                    'end' => Carbon::parse("{$this->end_date} {$this->end_time}")->format('F j, Y \a\t g:i A'),
                ],
                'duration' => Carbon::parse("{$this->start_date} {$this->start_time}")->diffForHumans(
                    Carbon::parse("{$this->end_date} {$this->end_time}"), 
                    true
                ),
            ],
            
            // Status flags
            'status' => [
                'is_expired' => Carbon::parse($this->end_date)->isPast(),
                'is_going_on' => Carbon::parse($this->start_date)->isToday() || 
                                (Carbon::parse($this->start_date)->isPast() && 
                                 Carbon::parse($this->end_date)->isFuture()),
                'time_until' => Carbon::parse("{$this->start_date} {$this->start_time}")->isFuture() ? 
                                Carbon::parse("{$this->start_date} {$this->start_time}")->diffForHumans() : null,
                'status' => $this->status ?? 'active',
            ],
            
            // Media and links
            'media' => [
                'cover_image' => getMedia($this->cover_image,'cover'),
                'event_url' => $this->event_url,
            ],
            
            // Event settings
            'settings' => [
                'type' => $this->type,
                'category' => $this->category,
                'allow_guests' => (bool)($this->allow_guests ?? false),
                'require_approval' => (bool)($this->require_approval ?? false),
                'max_participants' => $this->max_participants,
                'available_spots' => $this->max_participants ? 
                                    ($this->max_participants - $this->participants()->where('status', 'going')->count()) : 
                                    null,
            ],
            
            // User's relationship to event
            'user_status' => [
                'is_organizer' => $this->user_id === $userId,
                'is_going' => $this->isGoing($userId),
                'is_interested' => $this->isInterested($userId),
                'participation_status' => $this->participants()
                                        ->where('user_id', $userId)
                                        ->first()
                                        ?->status ?? null,
                'is_approved' => $this->participants()
                                ->where('user_id', $userId)
                                ->first()
                                ?->approved ?? true,
            ],
            
            // Organizer information
            'organizer' => new UserShortResource($this->whenLoaded('organizer')),
            
            // Participation stats
            'stats' => [
                'total_participants' => $this->participants()->count(),
                'going_count' => $this->participants()->where('status', 'going')->count(),
                'interested_count' => $this->participants()->where('status', 'interested')->count(),
                'is_full' => $this->max_participants ? 
                            ($this->participants()->where('status', 'going')->count() >= $this->max_participants) : 
                            false,
            ],
            
            // Participants (if loaded)
            'participants' => UserShortResource::collection(
                $this->whenLoaded('participants', 
                    function() {
                        return $this->participants->where('status', 'going')->map->user;
                    }
                )
            ),
            
            // Meta
            'created_at' => $this->created_at->toISOString(),
            'updated_at' => $this->updated_at->toISOString(),
        ];
    }
} 