<?php

namespace App\Http\Controllers;
use Stripe\Stripe;
use App\Models\User;
use App\Models\Deposit;
use App\Models\Package;
use App\Models\Withdraw;
use Stripe\PaymentIntent;
use App\Models\Transaction;
use App\Models\UserPackage;
use Illuminate\Http\Request;
use PayPalHttp\HttpException;
use App\Models\WithdrawRequest;
use App\Services\PayPalService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Flutterwave\Payments\Data\Currency;
use Illuminate\Support\Facades\Validator;
use PayPalHttp\HttpClient as PaypalClient;
use Flutterwave\Payments\Facades\Flutterwave;
use Stripe\Exception\InvalidRequestException;
use PayPalCheckoutSdk\Orders\OrdersGetRequest;
use Stripe\Checkout\Session as CheckoutSession;
use PayPalCheckoutSdk\Core\ProductionEnvironment as PaypalProdEnv;
use PayPalCheckoutSdk\Core\SandboxEnvironment as PaypalSandboxEnv;
use PayPalCheckoutSdk\Orders\OrdersCreateRequest as PaypalCreateOrder;
use PayPalCheckoutSdk\Orders\OrdersCaptureRequest as PaypalCaptureOrder;

class UserWalletController extends Controller
{
    /**
     * Display wallet main page
     */
    public function index(Request $request)
    {
        $user = $request->user();
        
        // Create wallet object for consistency with the view
        $wallet = (object)[
            'balance' => getUserWallet($user->id),
            'pending_balance' => 0, // You can implement this if needed
        ];

        // Get wallet statistics
        $stats = [
            'total_earned' => Transaction::where('user_id', $user->id)
            ->where('type', 'credit')
                ->sum('amount'),
                
            'monthly_earned' => Transaction::where('user_id', $user->id)
                ->where('type', 'credit')
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->sum('amount'),
                
            'total_withdrawn' => WithdrawRequest::where('user_id', $user->id)
                ->whereIn('status', [2, 3]) // Approved status
                ->sum('amount')
        ];

        // Get recent transactions
        $recentTransactions = Transaction::where('user_id', $user->id)
            ->latest()
            ->take(5)
            ->get();
        
        // Get pending withdrawals
        $pendingWithdrawals = WithdrawRequest::where('user_id', $user->id)
            ->whereIn('status', [1]) // Pending status
            ->latest()
            ->take(5)
            ->get();
        
        // Get most rewarded actions
        $topEarningActions = Transaction::where('user_id', $user->id)
            ->where('type', 'credit')
            ->whereNotNull('reference_type')
            ->selectRaw('reference_type, SUM(amount) as total_amount, COUNT(*) as count')
            ->groupBy('reference_type')
            ->orderBy('total_amount', 'desc')
            ->take(5)
            ->get();

        return view('front_end.pages.wallet.wallet', compact(
            'wallet', 
            'stats', 
            'recentTransactions', 
            'pendingWithdrawals', 
            'topEarningActions'
        ));
    }

    // Provider-specific deposit pages for front_end theme
    public function stripeDeposit(Request $request)
    {
        if (setting('stripe_payment', 1) != 1) {
            return redirect()->route('wallet.index')->with('error', 'Stripe deposits are currently disabled.');
        }

        return view('front_end.pages.wallet.deposit_stripe');
    }

    public function stripeCreateCheckout(Request $request)
    {
        if (setting('stripe_payment', 1) != 1) {
            return back()->with('error', 'Stripe payments are currently disabled.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:1',
        ]);
        $stripeSecret = setting('stripe_secret_key', env('STRIPE_SECRET'));
        if (blank($stripeSecret)) {
            return back()->with('error', 'Stripe secret key is missing. Please configure it in payment settings.');
        }
        Stripe::setApiKey($stripeSecret);
        $session = CheckoutSession::create([
            'payment_method_types' => ['card'],
            'mode' => 'payment',
            'line_items' => [[
                'price_data' => [
                    'currency' => 'usd',
                    'product_data' => [
                        'name' => 'Wallet top-up',
                    ],
                    'unit_amount' => (int) round($request->amount * 100),
                ],
                'quantity' => 1,
            ]],
            'metadata' => [
                'user_id' => $request->user()->id,
                'purpose' => 'wallet_deposit',
            ],
            'success_url' => route('wallet.deposit.stripe.success') . '?session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('wallet.deposit.stripe.cancel'),
        ]);
        return redirect()->away($session->url, 303);
    }

    public function stripeSuccess(Request $request)
    {
        $sessionId = $request->query('session_id');
        if (!$sessionId) {
            return redirect()->route('wallet.index')->with('error', 'Missing session reference');
        }
        $stripeSecret = setting('stripe_secret_key', env('STRIPE_SECRET'));
        Stripe::setApiKey($stripeSecret);
        $session = CheckoutSession::retrieve($sessionId);
        if ($session && $session->payment_status === 'paid') {
            $amount = ($session->amount_total ?? 0) / 100;
            $paymentIntentId = $session->payment_intent ?? $sessionId;
            $this->depositAmountsocialinkWallet($request->user()->id, $amount, 2, $paymentIntentId, strtoupper($session->currency ?? 'USD'));
            return redirect()->route('wallet.index')->with('success', 'Deposit successful via Stripe');
        }
        return redirect()->route('wallet.index')->with('error', 'Payment not completed');
    }
   

    public function stripeCancel()
    {
        return redirect()->route('wallet.index')->with('error', 'Stripe payment was cancelled');
    }

    public function paypalDeposit()
    {
        return view('front_end.pages.wallet.deposit_paypal');
    }

    public function paypalCreateCheckout(Request $request,PayPalService $paypal)
    {
         $validated = $request->validate([
            'amount' => ['required','numeric','min:0.01'],
        ]);

        $amount = (float) $validated['amount'];

        try {
            $order = $paypal->createOrder(
                amount: $amount,
                referenceId: 'wallet_deposit_'.($request->user()->id ?? 'guest'),
                returnUrl: route('wallet.deposit.paypal.success'),
                cancelUrl: route('wallet.deposit.paypal.cancel'),
            );

            // Find the approval link
            $approveUrl = collect($order->links ?? [])->firstWhere('rel', 'approve')->href ?? null;

            if (!$approveUrl) {
                Log::error('PayPal approve link missing', ['order' => $order]);
                return back()->with('error', 'Unable to start PayPal checkout.');
            }

            // (Optional) store for later reconciliation
            session([
                'paypal_order_id' => $order->id ?? null,
                'paypal_amount'   => $amount,
            ]);

            return redirect()->away($approveUrl);
        } catch (\Throwable $e) {
            Log::error('PayPal create order failed', ['e' => $e->getMessage()]);
            return back()->with('error', 'Unable to create PayPal order.');
        }

    }
    public function getPyapalTransactioDetails($transactionId)
    {
        $clientId = env('PAYPAL_CLIENT_ID');
        $secret = env('PAYPAL_CLIENT_SECRET');
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api-m.sandbox.paypal.com/v1/oauth2/token");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_USERPWD, $clientId . ":" . $secret);
        curl_setopt($ch, CURLOPT_POSTFIELDS, "grant_type=client_credentials&scope=https://uri.paypal.com/services/payments/realtimepayment");


        // Setting the headers
        $headers = [];
        $headers[] = "Accept: application/json";
        $headers[] = "Accept-Language: en_US";
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $response = curl_exec($ch);
        $accessToken = json_decode($response)->access_token;
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://api-m.sandbox.paypal.com/v1/payments/payment/" . $transactionId);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

        // Set the headers, including the Bearer token
        $headers = [];
        $headers[] = "Content-Type: application/json";
        $headers[] = "Authorization: Bearer " . $accessToken;
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        // Execute cURL request and store the response
        $response = curl_exec($ch);

        // Check for cURL errors
        if (curl_errno($ch)) {
            echo 'Error:' . curl_error($ch);
        } else {
            // Decode and display the transaction details
            $transactionDetails = json_decode($response, true);

            // Check if the transaction details were successfully retrieved
            if (isset($transactionDetails['id'])) {
                return $transactionDetails;
            }
        }

        curl_close($ch);
    }

    public function paypalSuccess(Request $request, PayPalService $paypal)
    {
        $orderId = $request->query('token'); // PayPal sends token=ORDER_ID
       

        try {
            // Optional: check status before capture
            $order = $paypal->getOrder($orderId); // APPROVED/CREATED/COMPLETED
        $result = $paypal->captureOrder($orderId);

        if (($result->status ?? null) === 'COMPLETED') {
          

            $capture = $result->purchase_units[0]->payments->captures[0] ?? null;

            $amount   = $capture->amount->value ?? null; // "10.00"
            $fee      = $capture->seller_receivable_breakdown->paypal_fee->value ?? null; // "0.69"

            $this->depositAmountsocialinkWallet($request->user()->id, $amount-$fee, 2, $result->id ?? null, 'USD');
            
            return redirect()->route('wallet.deposit.paypal')
                ->with('success', 'Amount deposit successfully');
        
        }
;

        } catch (HttpException $e) {
        
            return redirect()->route('wallet.deposit.paypal')
                ->with('error', 'PayPal authentication failed during capture. Check MODE & credentials.');
        } catch (\Throwable $e) {
            return redirect()->route('wallet.deposit.paypal')
                ->with('error', 'Unable to capture PayPal order.'.$e->getMessage());
        }
    }

    public function paypalCancel()
    {
        return redirect()->route('wallet.index')->with('error', 'PayPal payment was cancelled');
    }

    public function paypalTestCredentials()
    {
        $mode = setting('paypal_mode', env('PAYPAL_MODE', 'sandbox'));
        $clientId = trim((string) setting('paypal_id', env('PAYPAL_CLIENT_ID', '')));
        $clientSecret = trim((string) setting('paypal_secret', env('PAYPAL_CLIENT_SECRET', '')));
        
        $data = [
            'mode' => $mode,
            'client_id_present' => !empty($clientId),
            'client_secret_present' => !empty($clientSecret),
            'env_paypal_mode' => env('PAYPAL_MODE'),
            'env_paypal_client_id' => env('PAYPAL_CLIENT_ID') ? 'yes' : 'no',
            'env_paypal_client_secret' => env('PAYPAL_CLIENT_SECRET') ? 'yes' : 'no',
            'client_id_length' => strlen($clientId),
            'client_secret_length' => strlen($clientSecret),
            'client_id_start' => substr($clientId, 0, 10) . '...',
            'client_secret_start' => substr($clientSecret, 0, 10) . '...',
        ];
        
        // Test environment creation
        try {
            if ($mode === 'live') {
                $environment = new PaypalProdEnv($clientId, $clientSecret);
            } else {
                $environment = new PaypalSandboxEnv($clientId, $clientSecret);
            }
            $data['environment_created'] = true;
            $data['environment_class'] = get_class($environment);
        } catch (\Throwable $e) {
            $data['environment_created'] = false;
            $data['environment_error'] = $e->getMessage();
        }
        
        return response()->json($data);
    }

    public function flutterwaveDeposit()
    {
        if (setting('flutterwave_payment', 0) != 1) {
            return redirect()->route('wallet.index')->with('error', 'Flutterwave deposits are currently disabled.');
        }

        return view('front_end.pages.wallet.deposit_flutterwave');
    }

    public function flutterwaveCreateCheckout(Request $request)
    {
        if (setting('flutterwave_payment', 0) != 1) {
            return back()->with('error', 'Flutterwave payments are currently disabled.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:1',
        ]);
        
        $amount = $request->input('amount');

        $secretKey = setting('flutterwave_secret_key', env('FLW_SECRET_KEY'));
        if (blank($secretKey)) {
            return back()->with('error', 'Flutterwave secret key is missing. Please configure it in payment settings.');
        }

        $data = [
            "tx_ref" => uniqid("tx_"),
            "amount" => $amount,
            "currency" => "USD",
            "redirect_url" => route('wallet.deposit.flutterwave.callback'),
            "payment_options" => "card,banktransfer,ussd", // optional
            "customer" => [
                "email" => $request->user()->email ?? '',
                "name"  => $request->user()->name ?? '',
            ],
            "customizations" => [
                "title" => "Wallet Deposit",
                "description" => "Deposit to your wallet",
            ],
        ];


        $response = Http::withToken($secretKey)
            ->post('https://api.flutterwave.com/v3/payments', $data);

        $body = $response->json();

        // Debug log to storage/logs/laravel.log
        Log::info('Flutterwave response:', $body);

        if (isset($body['data']['link'])) {
            return redirect()->away($body['data']['link']);
        }

        return back()->with('error', 'Unable to initiate payment: ' . ($body['message'] ?? 'Unknown error'));
    }
 

    public function flutterwaveCallback(Request $request)
    {
        if (setting('flutterwave_payment', 0) != 1) {
            return redirect()->route('wallet.index')->with('error', 'Flutterwave payments are currently disabled.');
        }

        $transaction_id = $request->transaction_id;
        // $request->validate(['tx_ref' => 'required', 'amount' => 'required|numeric']);
            $secretKey = setting('flutterwave_secret_key', env('FLW_SECRET_KEY'));
            if (blank($secretKey)) {
                return redirect()->route('wallet.index')->with('error', 'Flutterwave secret key is missing. Please configure it in payment settings.');
            }

            $response = Http::withToken($secretKey)
        ->get("https://api.flutterwave.com/v3/transactions/{$transaction_id}/verify");

        $verification = $response->json();
        if($verification['data']['status'] == 'successful')
        {
            $amount = $verification['data']['amount_settled'];
            $this->depositAmountsocialinkWallet($request->user()->id, $amount, 3, $transaction_id, 'USD');
            return redirect()->route('wallet.index')->with('success', 'Deposit successful via Flutterwave');
        }
    }

    public function paystackDeposit()
    {
        if (setting('paystack_payment', 0) != 1) {
            return redirect()->route('wallet.index')->with('error', 'Paystack deposits are currently disabled.');
        }

        return view('front_end.pages.wallet.deposit_paystack');
    }

    public function paystackCreateCheckout(Request $request)
    {
        if (setting('paystack_payment', 0) != 1) {
            return response()->json([
                'message' => 'Paystack payments are currently disabled.'
            ], 422);
        }

        $request->validate(['amount' => 'required|numeric|min:1']);

        $secretKey = setting('paystack_secret_key');
        if (blank($secretKey)) {
            return response()->json([
                'message' => 'Paystack secret key is missing. Please configure it in payment settings.'
            ], 422);
        }

        // Placeholder redirect url
        $url = route('wallet.deposit.paystack.callback', ['reference' => uniqid(), 'amount' => $request->amount]);
        return response()->json(['redirect' => $url]);
    }

    public function paystackCallback(Request $request)
    {
        if (setting('paystack_payment', 0) != 1) {
            return redirect()->route('wallet.index')->with('error', 'Paystack payments are currently disabled.');
        }

        $request->validate(['reference' => 'required', 'amount' => 'required|numeric']);
        $this->depositAmountsocialinkWallet($request->user()->id, $request->amount, 4, $request->reference, 'USD');
        return redirect()->route('wallet.index')->with('success', 'Deposit successful via Paystack');
    }
	/**
	 * Get users for Select2 (exclude current user)
	 */
	public function getUsers(Request $request)
	{
		$currentUser = $request->user();
		$searchTerm = trim((string) $request->input('q', ''));
		$page = max(1, (int) $request->input('page', 1));
		$perPage = 20;

		$query = User::query()
			->where('id', '!=', $currentUser->id);

		if ($searchTerm !== '') {
			$query->where(function ($q) use ($searchTerm) {
				$q->where('name', 'like', "%{$searchTerm}%")
					->orWhere('email', 'like', "%{$searchTerm}%");
			});
		}

		$offset = ($page - 1) * $perPage;
		$users = $query
			->orderBy('name')
			->skip($offset)
			->take($perPage + 1)
			->get(['id', 'name', 'email']);

		$hasMore = $users->count() > $perPage;
		if ($hasMore) {
			$users = $users->slice(0, $perPage);
		}

		$results = $users->map(function ($user) {
			$label = trim(($user->name ?? 'Unknown') . ' (' . ($user->email ?? 'no-email') . ')');
			return [
				'id' => $user->id,
				'text' => $label,
			];
		})->values();

		return response()->json([
			'results' => $results,
			'pagination' => ['more' => $hasMore],
		]);
	}
    public function depositAmount(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'transaction_id' => 'required',
            'gateway_id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 400, 'message' => $validator->errors()->first()], 200);
        }
        $user = $request->user();
        $transactionId = $request->input('transaction_id');
        $gateway_id = $request->gateway_id;
        if($gateway_id == 1)
        {
            $stripeSecret = setting('stripe_secret_key', env('STRIPE_SECRET'));
            Stripe::setApiKey($stripeSecret);

            try {
                $data = PaymentIntent::retrieve($transactionId);
                
                if ($data->status == 'succeeded') {
                    $amount = $data->amount / 100;
                    $this->depositAmountsocialinkWallet($user->id,$amount,$gateway_id,$transactionId,'USD');
                    return response()->json(['status' => 200, 'message' => 'Amount deposited  Successfully'], 200);
                }
            } catch (InvalidRequestException $e) {
                // Handle any errors
                return response()->json(['error' => $e->getMessage()], 500);
            }
            return response()->json(['status' => '400', 'message' => '$'], 200);
        }
        elseif ($gateway_id == 2) {
            $transaction  = $this->getPyapalTransactioDetails($transactionId);
            $transactionFee = $transaction['transactions'][0]['related_resources'][0]['sale']['transaction_fee']['value'];
            $amount = $transaction['transactions'][0]['amount']['total'];
            $deposited_amount = $amount - $transactionFee;
            $this->depositAmountsocialinkWallet($user->id,$deposited_amount,$gateway_id,$transactionId,'USD');
            return response()->json(['status' => 200, 'message' => 'Amount deposited Successfully'], 200);
        }
        elseif($gateway_id==3){
            $transaction = $this->depositUsingFlutterwave($transactionId);
            $deposited_amount = $transaction['amount'] - $transaction['app_fee'] - $transaction['merchant_fee'];
            if ($transaction['currency'] != 'USD') {
                $deposited_amount = $this->convertCurrency($transaction['currency'], $deposited_amount);
               
            }
            $this->depositAmountsocialinkWallet($user->id,$deposited_amount,$gateway_id,$transactionId,'USD');
            return response()->json(['status' => 200, 'message' => 'Amount deposited Successfully'], 200);
        }
        

    }
    public function convertCurrency($fromCurrency, $amount)
    {
        $apiKey = '801a81b7c0f943322a0c920e';
        $toCurrency = 'USD';
        $url = "https://v6.exchangerate-api.com/v6/$apiKey/pair/$fromCurrency/$toCurrency";

        // Initialize cURL session
        $curl = curl_init();

        // Set the options for the cURL session
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
        ]);

        // Execute the request and store the response
        $response = curl_exec($curl);

        // Close the cURL session
        curl_close($curl);

        // Decode the response
        $data = json_decode($response, true);

        // Check if the request was successful
        if (isset($data['conversion_rate'])) {
            // Get the conversion rate
            $conversionRate = $data['conversion_rate'];

            // Calculate the converted amount
            $convertedAmount = $amount * $conversionRate;

            return $convertedAmount;
        } else {
            echo "Error retrieving conversion rate.\n";
        }
    }
     public function depositUsingFlutterwave($transaction_id)
    {
        $secret_key = env('FLW_SECRET_KEY');
        $url = "https://api.flutterwave.com/v3/transactions/$transaction_id/verify";
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer $secret_key",
            "Content-Type: application/json"
        ]);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            echo 'Error:' . curl_error($ch);
        } else {
            $result = json_decode($response, true);
            if ($result['status'] === 'success') {
                return $result['data'];
            } else {
                echo "Error: " . $result['message'];
            }
        }

        // Close cURL session
        curl_close($ch);
    }
    
    public function depositAmountsocialinkWallet($user_id,$amount,$withdraw_method_id,$transactionId,$currency='USD')
    {
        $deposit  = new Deposit();
        $deposit->user_id = $user_id;
        $deposit->withdraw_method_id = $withdraw_method_id;
        $deposit->amount = $amount;
        $deposit->currency = strtoupper($currency);
        $deposit->charge = 0;
        $deposit->final_amo = $amount;
        $deposit->deposit_status = 1;
        $deposit->transaction_id = $transactionId;
        $deposit->save();

        $transaction = new Transaction;
        $transaction->wallet_id = 1;
        
        $transaction->user_id = $user_id;
        $transaction->type = 'credit';
        $transaction->reference_type = 'amount_deposit';
        $transaction->reference_id = $deposit->id;
        $transaction->amount = $amount;
        $transaction->description = 'amount deposited';
        $transaction->save();
        // $notification = new Notification();
        // $notification->to_user_id = $user_id; // User ID of the requester
        // $notification->text = "approved your deposite request Successfully ";
        // $notification->notification_type = "deposite";
        // $notification->save();
    }
    public function transferAmount(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'to_userId' => 'required',
            'amount' => 'required',
            'password'=>'required'
        ]);

        if ($validator->fails()) {
            return response()->json(['status'=>400,'message' => $validator->errors()->first()], 200);
        }
        $to_userId = $request->to_userId;
        $user = $request->user();
        if (!Hash::check($request->password, $request->user()->password)) {
            return response()->json(['status'=>400,'message' => 'The password  is incorrect.'], 200);
        }
        $amount = $request->amount;

        $userwallet = getUserWallet($user->id);
        if ($amount > $userwallet) {
            return response()->json(['status'=>400,'message' => 'insufficient balance, Available balance is '.$userwallet, ], 200);
        }
        if($user->level==1)
        {
            return response()->json(['status'=>400,'message' => 'Please upgrade your package to transfer '], 200);
        }
        $nextuser  = User::find($to_userId);
        if(empty($nextuser))
        {
            return response()->json(['status'=>400,'message' => 'user not found ',  ], 200);
        }

        if($to_userId==$user->id)
        {
            return response()->json(['status'=>400,'message' => 'can not transfer amount yourself ',  ], 200);
        }
        $transaction = new Transaction();
        $transaction->user_id = $user->id;
        $transaction->flag = 'debit';
        $transaction->amount = $amount;
        $transaction->description = 'amount Transfer';
        $transaction->save();
        $transaction = new Transaction();
        $transaction->user_id = $to_userId;
        $transaction->flag = 'credit';
        $transaction->amount = $amount;
        $transaction->description = 'Amount Receieved';
        $transaction->save();
        return response()->json(['status' =>200,'message'=>'Balance transfer successfully'], 200);
    }
    public function withdrawHistory(Request $request)
    {
        $user = $request->user();
        $offset = !empty($request->offset)?$request->offset:0;
        $limit = !empty($request->limit)?$request->limit:10;

        $withdraw_requests = WithdrawRequest::where('user_id',$user->id)->orderBy('id','desc')->limit($limit)->offset($offset)->get();
        if(!empty($withdraw_requests))
        {
            $statusArr = [
                1=>'Pending',
                2=>'Approved',
                3=>'Approved'

            ];
            foreach($withdraw_requests as $withdraw_request)
            {
                $withdraw_request['status'] = $statusArr[$withdraw_request->status];
            }
            return response()->json([
                'status' => 200,
                'message' => 'Withdraw history fetch successfully',
                'data'=>$withdraw_requests
            ]);
        }
        return response()->json([
            'status' => 200,
            'message' => 'Withdraw History Not found',
            'data'=>$withdraw_requests
        ]);
    }
    public function transactionHistory(Request $request)
    {
        {
            $user = $request->user();
            $offset = !empty($request->offset)?$request->offset:0;
            $limit = !empty($request->limit)?$request->limit:10;
            $transactionHistory = Transaction::where(['user_id'=>$user->id])->orderBy('id','desc')->limit($limit)->offset($offset)->get();
            if(!empty($transactionHistory))
            {

                return response()->json([
                    'status' => 200,
                    'message' => 'Transaction History fetch successfully',
                    'data'=>$transactionHistory
                ]);
            }
            return response()->json([
                'status' => 200,
                'message' => 'Transaction History Not found',
            ]);
        }
    }
    public function depositHistory(Request $request)
    {
          $user = $request->user();
        // print_r($user);
        $limit = $request->limit??10;
        $offset = $request->offset??0;
        $deposits = Deposit::where('user_id',$user->id)->limit($limit)->offset($offset)->get();
        $withdrawMethod = [
            1=>'Stripe',
            2=>'Paypal',
            3=>'Flutterwave',
            4=>'Paystack'
        ];
        $statusArray = [
            0=>'pending',
            1=>'success',
            2=>'cancel'
        ];

        if(count($deposits)>0)
        {

            foreach($deposits as $deposit)
            {
                $deposit['withdraw_method'] = $withdrawMethod[$deposit->withdraw_method_id];
                $deposit['deposit_status'] = $statusArray[$deposit->deposit_status]??'';
            }
        }
        return response()->json(['status' => 200,'message'=>'Deposit history fetch successfully', 'data' => $deposits], 200);

    }
    public function createWithdrawRequest(Request $request)
    {
        $rules = [
            'type'   => 'required|in:Paypal,Bank',
            'amount' => 'required|numeric|gt:0',
        ];
    
        $messages = [
            'type.required'   => __('api.withdraw_type_required'),
            'amount.required' => __('api.withdraw_amount_required'),
            'amount.gt'       => __('api.withdraw_amount_gt0'),
        ];
    
        $validator = Validator::make($request->all(), $rules, $messages);
    
        if ($validator->fails()) {
            return response()->json([
                'status'  => 400,
                'message' => $validator->errors()->first(),
            ], 200);
        }
    
        $user = auth()->user();
    
        if ($user->id == 1) {
            return response()->json([
                'status'  => 400,
                'message' => __('api.admin_withdraw_error'),
            ], 200);
        }
 
        $total_amount    = getUserWallet($user->id); // helper to fetch wallet balance
        $request_amount  = $request->input('amount');
        $type            = $request->input('type');
        if($type=='Paypal' && setting('paypal_withdraw')==0){
            return response()->json([
                'status'  => 400,
                'message' => __('paypal withdrawal is not enabled'),
            ], 200);
        }
        if($type=='Bank' && setting('bank_withdraw')==0){
            return response()->json([
                'status'  => 400,
                'message' => __('bank withdrawal is not enabled'),
            ], 200);
        }
       
        if ($request_amount > $total_amount) {
            return response()->json([
                'status'  => 400,
                'message' => __('api.insufficient_balance'),
            ], 200);
        }
        if($request_amount < setting('m_withdrawal')){
            return response()->json([
                'status'  => 400,
                'message' => __('minimum withdrawal amount is ' . setting('m_withdrawal')),
            ], 200);
        }
    
        if ($type === 'Paypal') {
            $paypalEmail = $request->input('paypal_email');
    
            if (empty($paypalEmail)) {
                return response()->json([
                    'status'  => 400,
                    'message' => __('api.paypal_email_required'),
                ], 200);
            }
    
            $withdraw = new Withdraw();
            $withdraw->type         = 'Paypal';
            $withdraw->user_id      = $user->id;
            $withdraw->amount       = $request_amount;
            $withdraw->paypal_email = $paypalEmail;
            $withdraw->save();
    
            // add transaction
            $this->addTransaction($user, $request_amount, $withdraw);
    
            return response()->json([
                'status'  => 200,
                'message' => __('api.paypal_withdraw_success'),
            ], 200);
    
        } elseif ($type === 'Bank') {
            $iban       = $request->input('iban');
            $address    = $request->input('address');
            $country    = $request->input('country');
            $swift_code = $request->input('swift_code');
            $full_name  = $request->input('full_name');
    
            $fieldsToCheck = compact('iban', 'country', 'swift_code', 'full_name', 'address');
            $emptyFields   = array_keys(array_filter($fieldsToCheck, fn($value) => empty($value)));
    
            if (!empty($emptyFields)) {
                $errorMessage = __('api.missing_fields') ." ". implode(', ', $emptyFields);
                return response()->json([
                    'status'  => 400,
                    'message' => $errorMessage,
                ], 200);
            }
    
            $withdraw = new WithdrawRequest();
            $withdraw->type       = 'Bank';
            $withdraw->user_id    = $user->id;
            $withdraw->amount     = $request_amount;
            $withdraw->iban       = $iban;
            $withdraw->country    = $country;
            $withdraw->swift_code = $swift_code;
            $withdraw->full_name  = $full_name;
            $withdraw->address    = $address;
            $withdraw->save();
    
            // add transaction
            $this->addTransaction($user, $request_amount, $withdraw);
    
            return response()->json([
                'status'  => 200,
                'message' => __('api.bank_withdraw_success'),
            ], 200);
        }
    }
    
    /**
     * Handle transaction creation
     */
    protected function addTransaction($user, $amount, $withdraw)
    {
        // Debit user
        $transaction = new Transaction();
        $transaction->user_id        = $user->id;
        $transaction->wallet_id      = walletId();
        $transaction->amount         = $amount;
        $transaction->type           = 'debit';
        $transaction->description    = 'Withdraw';
        $transaction->reference_type = 'withdraw';
        $transaction->reference_id   = $withdraw->id;
        $transaction->save();
    
        // Credit admin (id=1)
        $transaction = new Transaction();
        $transaction->user_id        = 1;
        $transaction->wallet_id      = walletId();
        $transaction->amount         = $amount;
        $transaction->type           = 'credit';
        $transaction->description    = 'Withdraw';
        $transaction->reference_type = 'withdraw';
        $transaction->reference_id   = $withdraw->id;
        $transaction->save();
    }
    
    public function getUserWallet(Request $request)
    {
        $user = $request->user();
        $data['balance'] = getUserWallet($user->id);
        $data['last_withdraw'] = (float) (WithdrawRequest::select('amount')
        ->where('user_id', $user->id)
        ->where('status', 2)
        ->latest()
        ->first()->amount ?? 0);
    
        $data['last_depoist'] = (float) (Deposit::select('amount')
            ->where(['user_id' => $user->id, 'deposit_status' => 1])
            ->latest()
            ->first()->amount ?? 0);
        return response()->json([
            'status' => 200,
            'message' => 'Wallet fetch successfully',
            'data' => $data
        ]);
    }
    public function upgradeToPro(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'package_id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 400, 'message' => $validator->errors()->first()], 200);
        }

        $user = $request->user();
        $package_id = $request->package_id;
        $package = Package::find($package_id);
        if(!$package)
        {
            return response()->json(['status' => 400, 'message' => 'package not found '], 200);
        }
        if($user->level>$package_id){
            return response()->json(['status' => 400, 'message' => 'Already upgraded package subscribed'], 200);
        }
        $balance = getUserWallet($user->id);
        if($balance<$package->package_price)
        {
            return response()->json(['status' => 400, 'message' => 'Insufficent Balance'], 200);
        }
       // Debit user
        $userPackage = new UserPackage();
        $userPackage->user_id = $user->id;
        $userPackage->package_id = $package_id;
        $userPackage->subscription_date = now()->toDateString();
        $userPackage->expires_at = now()->addMonth()->toDateString();
        $userPackage->save();
        $user->level=$package->id;
        $user->update();
        $transaction = new Transaction();
        $transaction->user_id        = $user->id;
        $transaction->wallet_id      = walletId();
        $transaction->amount         = $package->package_price;
        $transaction->type           = 'debit';
        $transaction->description    = 'Package subscribed';
        $transaction->reference_type = 'package_subscribed';
        $transaction->reference_id   = $userPackage->id;
        $transaction->save();
        $transaction = new Transaction();
        $transaction->user_id        = 1;
        $transaction->wallet_id      = walletId();
        $transaction->amount         = $package->package_price;
        $transaction->type           = 'credit';
        $transaction->description    = 'Package subscribed';
        $transaction->reference_type = 'package_subscribed';
        $transaction->reference_id   = $userPackage->id;
        $transaction->save();
        return response()->json(['status' => 200, 'message' => 'Package subscribed successfully'], 200);

    }
    public function upgradeToProWeb(Request $request)
    {
        $user = $request->user();
        $packages = Package::all();
        return view('front_end.pages.wallet.upgrade_to_pro', compact('user', 'packages'));
    }
   
}



