<?php

namespace App\Http\Controllers;

use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\CourseResource;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class CourseController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'address' => 'required|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'category_id' => 'required|integer|exists:categories,id',
            'country' => 'required|string',
            'level' => 'required|integer',
            'language' => 'required|string',
            'description' => 'nullable|string',
            'is_paid' => 'required|boolean',
            'amount' => 'nullable|numeric|min:0',
            'cover' => 'nullable|image|max:2048',
        ]);

        $data = $request->all();
        $data['created_by'] = Auth::id();
        if ($request->hasFile('cover')) {
            $data['cover'] = $request->file('cover')->store('covers', 'public');
        }

        $course = Course::create($data);

        return response()->json([
            'status' => 201,
            'message' => 'Course created successfully.',
            'data' => new CourseResource($course),
        ], 201);
    }

    public function index(Request $request)
    {
        $courses = Course::with('user')
            ->paginate(10);

        return response()->json([
            'status' => 200,
            'message' => 'Courses retrieved successfully.',
            'data' => CourseResource::collection($courses),
        ]);
    }

    public function show($id)
    {
        try {
            $course = Course::with('user')->findOrFail($id);

            return response()->json([
                'status' => 200,
                'message' => 'Course retrieved successfully.',
                'data' => new CourseResource($course),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 404,
                'message' => 'Course not found.',
            ], 404);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $course = Course::findOrFail($id);

            $request->validate([
                'title' => 'nullable|string|max:255',
                'address' => 'nullable|string',
                'start_date' => 'nullable|date',
                'end_date' => 'nullable|date|after:start_date',
                'category_id' => 'nullable|integer|exists:categories,id',
                'country' => 'nullable|string',
                'level' => 'nullable|integer',
                'language' => 'nullable|string',
                'description' => 'nullable|string',
                'is_paid' => 'nullable',
                'amount' => 'nullable|numeric|min:0',
                'cover' => 'nullable|image|max:2048',
            ]);

            $data = $request->all();
            if ($request->hasFile('cover')) {
                $data['cover'] = $request->file('cover')->store('covers', 'public');
            }

            $course->update($data);

            return response()->json([
                'status' => 200,
                'message' => 'Course updated successfully.',
                'data' => new CourseResource($course),
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 404,
                'message' => 'Course not found.',
            ], 404);
        }
    }

    public function destroy($id)
    {
        try {
            $course = Course::findOrFail($id);

            $course->delete();

            return response()->json([
                'status' => 200,
                'message' => 'Course deleted successfully.',
            ]);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 404,
                'message' => 'Course not found.',
            ], 404);
        }
    }
}
