<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MediaController extends Controller
{
    /**
     * Update media settings
     */
    public function update(Request $request)
    {
        try {
            $settingsType = $request->input('settings_type');
            
            switch ($settingsType) {
                case 'image_processing':
                    return $this->updateImageProcessingSettings($request);
                case 'security':
                    return $this->updateSecuritySettings($request);
                case 'avatar_config':
                    return $this->updateAvatarConfig($request);
                case 'cover_config':
                    return $this->updateCoverConfig($request);
                case 'image_config':
                    return $this->updateImageConfig($request);
                case 'video_config':
                    return $this->updateVideoConfig($request);
                case 'document_config':
                    return $this->updateDocumentConfig($request);
                default:
                    return response()->json(['success' => false, 'message' => 'Invalid settings type']);
            }
        } catch (\Exception $e) {
            Log::error('Media settings update failed', [
                'error' => $e->getMessage(),
                'settings_type' => $request->input('settings_type')
            ]);
            
            return response()->json(['success' => false, 'message' => 'Failed to update media settings']);
        }
    }

    /**
     * Update image processing settings
     */
    private function updateImageProcessingSettings(Request $request)
    {
        $settings = [
            'media_image_processing' => $request->has('media_image_processing'),
            'media_image_quality' => $request->input('media_image_quality', 85),
            'media_resize_large_images' => $request->has('media_resize_large_images'),
            'media_create_thumbnails' => $request->has('media_create_thumbnails'),
        ];

        // Save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Image processing settings updated successfully']);
    }

    /**
     * Update security settings
     */
    private function updateSecuritySettings(Request $request)
    {
        $settings = [
            'media_scan_uploads' => $request->has('media_scan_uploads'),
            'media_allowed_mime_types' => $request->input('media_allowed_mime_types'),
            'media_blocked_extensions' => $request->input('media_blocked_extensions'),
        ];

        // Validate MIME types format
        if (!empty($settings['media_allowed_mime_types'])) {
            $mimeTypes = array_map('trim', explode(',', $settings['media_allowed_mime_types']));
            $settings['media_allowed_mime_types'] = implode(',', $mimeTypes);
        }

        // Validate blocked extensions format
        if (!empty($settings['media_blocked_extensions'])) {
            $extensions = array_map('trim', explode(',', $settings['media_blocked_extensions']));
            $settings['media_blocked_extensions'] = implode(',', $extensions);
        }

        // Save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Security settings updated successfully']);
    }

    /**
     * Update avatar configuration
     */
    private function updateAvatarConfig(Request $request)
    {
        $settings = [
            'avatar_max_size' => $request->input('avatar_max_size', 2048),
            'avatar_allowed_extensions' => $request->input('avatar_allowed_extensions', 'jpg,jpeg,png,gif'),
            'avatar_max_width' => $request->input('avatar_max_width', 500),
            'avatar_max_height' => $request->input('avatar_max_height', 500),
        ];

        // Validate and save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Avatar settings updated successfully']);
    }

    /**
     * Update cover image configuration
     */
    private function updateCoverConfig(Request $request)
    {
        $settings = [
            'cover_max_size' => $request->input('cover_max_size', 5120),
            'cover_allowed_extensions' => $request->input('cover_allowed_extensions', 'jpg,jpeg,png'),
            'cover_max_width' => $request->input('cover_max_width', 1920),
            'cover_max_height' => $request->input('cover_max_height', 1080),
        ];

        // Validate and save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Cover image settings updated successfully']);
    }

    /**
     * Update general image configuration
     */
    private function updateImageConfig(Request $request)
    {
        $settings = [
            'image_max_size' => $request->input('image_max_size', 10240),
            'image_allowed_extensions' => $request->input('image_allowed_extensions', 'jpg,jpeg,png,gif,webp'),
            'image_max_width' => $request->input('image_max_width', 4096),
            'image_max_height' => $request->input('image_max_height', 4096),
        ];

        // Validate and save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'General image settings updated successfully']);
    }

    /**
     * Update video configuration
     */
    private function updateVideoConfig(Request $request)
    {
        $settings = [
            'video_max_size' => $request->input('video_max_size', 102400),
            'video_allowed_extensions' => $request->input('video_allowed_extensions', 'mp4,avi,mov,wmv,flv'),
        ];

        // Validate and save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Video settings updated successfully']);
    }

    /**
     * Update document configuration
     */
    private function updateDocumentConfig(Request $request)
    {
        $settings = [
            'document_max_size' => $request->input('document_max_size', 51200),
            'document_allowed_extensions' => $request->input('document_allowed_extensions', 'pdf,doc,docx,txt,rtf'),
        ];

        // Validate and save settings
        foreach ($settings as $key => $value) {
            setting([$key => $value]);
        }

        return response()->json(['success' => true, 'message' => 'Document settings updated successfully']);
    }
} 